#include <string.h>
#include <stdlib.h>
#include "sqmrunner.h"
#include "SQMWorksetCInterface.h"

static int Initialise(SQMRunner* pObj, FILE* pOut, FILE* pErr);
static int AddRemoveVariables(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr);
static int AddRemoveObservations(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr);
static int PrintWorkset(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr);
static int PrintBlock(SQM_WorksetHandle hWorkset, SQM_VariableBlock eBlock, FILE* pOut, FILE* pErr);
static int PrintObservations(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr);
static int Cleanup(SQM_WorksetHandle hWorkset, SQMRunner* pObj, FILE* pOut, FILE* pErr);
static int SetScaling(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr);
static int SetTransform(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr);
static int SaveModel(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr);

int SQMRunner_Workset(SQMRunner* pObj, FILE* pOut, FILE* pErr)
{
   const char        *szErrString;           /* String that will contain possible error. */
   SQM_WorksetHandle hWorkset;               /* The worket handle */

   /* Open the project and set it in modeling mode */
   if (!Initialise(pObj, pOut, pErr))
      return 0;

   /* Get a new workset handle */
   if (!SQM_GetNewWorkset(pObj->mszProjID, &hWorkset))
   {
      szErrString="SQM_GetNewWorkset failed.";
      goto error_exit;
   }

   /* Create a default workset. */
   if (!SQMWS_CreateDefaultWorkset(hWorkset))
   {
      szErrString="SQMWS_CreateDefaultWorkset failed.";
      goto error_exit;
   }

   /* Adds and removes some observations and assigns them to classes */
   if (!AddRemoveObservations(hWorkset, pOut, pErr))
      return 0;

   /* Adds and removes some variables, assigns classes to variables and creates a Y-variable */
   if (!AddRemoveVariables(hWorkset, pOut, pErr))
      return 0;

   /* Sets scaling for some variables. */
   if (!SetScaling(hWorkset, pOut, pErr))
      return 0;

   /* Set transforms for some variables. */
   if (!SetTransform(hWorkset, pOut, pErr))
      return 0;

   /* Set name, description, missing value tolerans and save the model */
   if (!SaveModel(hWorkset, pOut, pErr))
      return 0;

   /* Print the resulting workset */
   if (!PrintWorkset(hWorkset, pOut, pErr))
      return 0;

   /* Release the workset and close the project */
   if (!Cleanup(hWorkset, pObj, pOut, pErr))
      return 0;
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

static int AddRemoveObservations(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr)
{
   char                 *szErrString;
   int                  iWorksetIndex;
   char                 *szName;
   int                  iNumObservations;
   int                  iClassNo;

   /* Remove the third and fourth observation */
   if (!SQMWS_GetObservationName(hWorkset, 3, 1, &szName))
   {
      szErrString="SQMWS_GetObservationName failed";
      goto error_exit;
   }
   fprintf(pOut, "Removing observation '%s' with index 3.\n", szName);
   if (!SQMWS_RemoveObservation(hWorkset, 3))
   {
      szErrString="SQMWS_RemoveObservation failed";
      goto error_exit;
   }
   /* Note that the fourth observation now has index 3 in the workset since we removed the previous */
   /* observation with index 3 */
   if (!SQMWS_GetObservationName(hWorkset, 3, 1, &szName))
   {
      szErrString="SQMWS_GetObservationName failed";
      goto error_exit;
   }
   fprintf(pOut, "Removing observation '%s' with index 3.\n", szName);
   if (!SQMWS_RemoveObservation(hWorkset, 3))
   {
      szErrString="SQMWS_RemoveObservation failed";
      goto error_exit;
   }
   /* Add the fourth observation back to the workset again. */
   /* Note that the index is four now since it is an index in the primary dataset. */
   if (!SQMWS_AddObservation(hWorkset, 4, &iWorksetIndex))
   {
      szErrString="SQMWS_AddObservation failed";
      goto error_exit;
   }
   if (!SQMWS_GetObservationName(hWorkset, iWorksetIndex, 1, &szName))
   {
      szErrString="SQMWS_GetObservationName failed";
      goto error_exit;
   }
   fprintf(pOut, "Adding observation '%s' with index %i.\n", szName, iWorksetIndex);
   /* Divide the observations in three classes */
   if (!SQMWS_GetObservationSize(hWorkset, &iNumObservations))
   {
      szErrString="SQMWS_GetObservationSize failed";
      goto error_exit;
   }
   for (iWorksetIndex=1; iWorksetIndex<=iNumObservations; ++iWorksetIndex)
   {
      if (iWorksetIndex < iNumObservations/3)
         iClassNo=1;
      else if (iWorksetIndex < 2*iNumObservations/3)
         iClassNo=2;
      else
         iClassNo=3;
      if (!SQMWS_SetObservationClass(hWorkset, iWorksetIndex, iClassNo))
      {
         szErrString="SQMWS_SetObservationClass failed";
         goto error_exit;
      }
   }
   /* Reset the first observation so that it doesn't belong to any class */
   if (!SQMWS_SetObservationClass(hWorkset, 1, 0))
   {
      szErrString="SQMWS_SetObservationClass failed";
      goto error_exit;
   }
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

int SaveModel(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr)
{
   char                 *szErrString;
   int                  iModelNumber;

   /* Set missing value tolerance */
   if (!SQMWS_SetMissingValueToleranceObs(hWorkset, 0.7f))
   {
      szErrString="SQMWS_SetMissingValueToleranceObs failed.";
      goto error_exit;
   }
   if (!SQMWS_SetMissingValueToleranceVar(hWorkset, 0.75f))
   {
      szErrString="SQMWS_SetMissingValueToleranceVar failed.";
      goto error_exit;
   }

   if (!SQMWS_SetWorksetDescription(hWorkset, "This model was created by SQMCSample.exe"))
   {
      szErrString="SQMWS_SetWorksetDescription failed.";
      goto error_exit;
   }

   /* Create a new model for the workset. */
   if (!SQMWS_CreateModel(hWorkset, &iModelNumber))
   {
      szErrString="SQMWS_CreateModel failed.";
      goto error_exit;
   }
   fprintf(pOut, "\nNew model %i created.\n", iModelNumber);

   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

int SetTransform(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr)
{
   SQMWS_TransformInfo  oTransform;
   char                 *szErrString;

   /* Set the transform of the first variable to 2*X-3 */
   oTransform.eTransformType=SQMWS_TransformLinear;
   oTransform.fConstant_a=2;
   oTransform.fConstant_b=-3;
   if (!SQMWS_SetTransform(hWorkset, SQM_BlockX, 1, &oTransform))
   {
      szErrString="SQMWS_SetTransform failed.";
      goto error_exit;
   }
   /* Set the transform of the Second variable to log10(x) */
   oTransform.eTransformType=SQMWS_TransformLog;
   oTransform.fConstant_a=1;
   oTransform.fConstant_b=0;
   if (!SQMWS_SetTransform(hWorkset, SQM_BlockX, 2, &oTransform))
   {
      szErrString="SQMWS_SetTransform failed.";
      goto error_exit;
   }
   /* Set the transform of the third variable to -log10(x) */
   oTransform.eTransformType=SQMWS_TransformNegLog;
   oTransform.fConstant_a=1;
   oTransform.fConstant_b=0;
   if (!SQMWS_SetTransform(hWorkset, SQM_BlockX, 3, &oTransform))
   {
      szErrString="SQMWS_SetTransform failed.";
      goto error_exit;
   }
   /* Set the transform of the fourth variable to log10((X-3)/(100-X))*/
   oTransform.eTransformType=SQMWS_TransformLogIt;
   oTransform.fConstant_a=3;
   oTransform.fConstant_b=100;
   if (!SQMWS_SetTransform(hWorkset, SQM_BlockX, 4, &oTransform))
   {
      szErrString="SQMWS_SetTransform failed.";
      goto error_exit;
   }
   /* Set the transform of the fifth variable to e^(0.0001*x+2) */
   oTransform.eTransformType = SQMWS_TransformExp;
   oTransform.fConstant_a = 0.0001f;
   oTransform.fConstant_b = 2;
   if (!SQMWS_SetTransform(hWorkset, SQM_BlockX, 5, &oTransform))
   {
      szErrString="SQMWS_SetTransform failed.";
      goto error_exit;
   }

   /* Set the transform of the sixth variable to the square root of (2*X+5)*/
   oTransform.eTransformType=SQMWS_TransformPower;
   oTransform.fConstant_a=2;
   oTransform.fConstant_b=5;
   oTransform.eExponent=SQMWS_ExpSquareRoot;
   if (!SQMWS_SetTransform(hWorkset, SQM_BlockX, 6, &oTransform))
   {
      szErrString="SQMWS_SetTransform failed.";
      goto error_exit;
   }
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

int SetScaling(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr)
{
   const char        *szErrString;           /* String that will contain possible error. */
   SQMWS_ScaleInfo   oScaleInfo;
   int               iNumXVars;
   SQX_IntVector     oDS1Indecies;
   SQX_IntVector     oLags;
   int iNumVars;
   int iLag;
   int               ix;

   /* Set pareto scaling with a fixed center value for variable 1 */
   oScaleInfo.eBaseScalingType=SQMWS_ScalePareto;
   oScaleInfo.eBlockScalingType=SQMWS_BlockScaleNone;
   oScaleInfo.eCentering=SQMWS_CenterFixed;
   oScaleInfo.fCenter = 1.1f;
   oScaleInfo.fScaleModifier=1;
   if (!SQMWS_SetScaling(hWorkset, SQM_BlockX, 1, &oScaleInfo))
   {
      szErrString="SQMWS_SetScaling failed.";
      goto error_exit;
   }

   /* Set a fixed scaling weight and auto centering for variable 2 */
   oScaleInfo.eBaseScalingType=SQMWS_ScaleFrozen;
   oScaleInfo.fScalingWeight = 5.4f;
   oScaleInfo.fCenter = 2.2f;
   oScaleInfo.eBlockScalingType=SQMWS_BlockScaleNone;
   oScaleInfo.fScaleModifier=1;
   if (!SQMWS_SetScaling(hWorkset, SQM_BlockX, 2, &oScaleInfo))
   {
      szErrString="SQMWS_SetScaling failed.";
      goto error_exit;
   }

   /* Change the modifier for variable 3 and set block scaling*/
   if (!SQMWS_GetScaling(hWorkset, SQM_BlockX, 3, &oScaleInfo))
   {
      szErrString="SQMWS_SetScaling failed.";
      goto error_exit;
   }
   oScaleInfo.fScaleModifier = 3.3f;
   oScaleInfo.eBlockScalingType=SQMWS_BlockScaleSquareRoot;
   oScaleInfo.iBlockScalingBlock=1;
   if (!SQMWS_SetScaling(hWorkset, SQM_BlockX, 3, &oScaleInfo))
   {
      szErrString="SQMWS_SetScaling failed.";
      goto error_exit;
   }

   /* Set blockscaling for variable 4 */
   if (!SQMWS_GetScaling(hWorkset, SQM_BlockX, 4, &oScaleInfo))
   {
      szErrString="SQMWS_SetScaling failed.";
      goto error_exit;
   }
   oScaleInfo.eBlockScalingType=SQMWS_BlockScaleSquareRoot;
   oScaleInfo.iBlockScalingBlock=1;
   if (!SQMWS_SetScaling(hWorkset, SQM_BlockX, 4, &oScaleInfo))
   {
      szErrString="SQMWS_SetScaling failed.";
      goto error_exit;
   }
   /* Set block scaling and modifier for expanded and laged variables */
   if (!SQMWS_GetVariableSize(hWorkset, SQM_BlockX, &iNumXVars))
   {
      szErrString="SQMWS_GetVariableSize failed.";
      goto error_exit;
   }
   for (ix=1; ix<=iNumXVars; ++ix)
   {
      if (!SQMWS_GetVariable(hWorkset, SQM_BlockX, ix, &oDS1Indecies, &oLags))
      {
         szErrString="SQMWS_GetVariable failed.";
         goto error_exit;
      }
      iNumVars=SQX_GetIntVectorSize(&oDS1Indecies);
      SQX_GetDataFromIntVector(&oLags,1,&iLag);
      if (iNumVars>1 || iLag>0)
      {
         if (!SQMWS_GetScaling(hWorkset, SQM_BlockX, ix, &oScaleInfo))
         {
            szErrString="SQMWS_GetScaling failed.";
            goto error_exit;
         }
         oScaleInfo.iBlockScalingBlock=1;
         oScaleInfo.eBlockScalingType=SQMWS_BlockScaleSquareRoot;
         oScaleInfo.fScaleModifier=2;
         if (!SQMWS_SetScaling(hWorkset, SQM_BlockX, ix, &oScaleInfo))
         {
            szErrString="SQMWS_SetScaling failed.";
            goto error_exit;
         }
      }
   }
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

int Cleanup(SQM_WorksetHandle hWorkset, SQMRunner* pObj, FILE* pOut, FILE* pErr)
{
   const char        *szErrString;           /* String that will contain possible error. */
   /* Release the handle */
   if (!SQMWS_ReleaseWorksetHandle(hWorkset))
   {
      szErrString="SQMWS_ReleaseWorksetHandle failed.";
      goto error_exit;
   }

   if (!SQX_RemoveProject(pObj->mszProjID))
   {
      szErrString="RemoveProject failed.";
      goto error_exit;
   }
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

static int PrintBlock(SQM_WorksetHandle hWorkset, SQM_VariableBlock eBlock, FILE* pOut, FILE* pErr)
{
   const char        *szErrString;           /* String that will contain possible error. */
   /* Various variables that holds temporary values */
   int                     iSize;
   int                     ix;
   SQX_IntVector            oClasses;
   int                     iClass;
   int                     iNumClasses;
   char                    *szName;
   SQMWS_ScaleInfo         oScaleInfo;
   SQMWS_TransformInfo     oTransformInfo;
   char                    szBuffer[100];

   /* Get the number of variables */
   if (!SQMWS_GetVariableSize(hWorkset, eBlock, &iSize))
   {
      szErrString="SQMWS_GetVariableSize failed.";
      goto error_exit;
   }
   fprintf(pOut, "\nNumber of %c-variables: %i.\n", eBlock==SQM_BlockX? 'X' : 'Y', iSize);
   fprintf(pOut, "Name                Class Scale     Long scaling\n");
   /* List the variables, scaling and class */
   for (ix=1; ix<=iSize; ++ix)
   {
      if (!SQMWS_GetVariableName(hWorkset, eBlock, ix, 1, &szName))
      {
         szErrString="SQMWS_GetVariableName failed.";
         goto error_exit;
      }
      fprintf(pOut, "%-20s", szName);
      if (!SQMWS_GetVariableClass(hWorkset, eBlock, ix, &oClasses))
      {
         szErrString="SQMWS_GetVariableClass failed.";
         goto error_exit;
      }
      szBuffer[0]='\0';
      iNumClasses=SQX_GetIntVectorSize(&oClasses);
      if (iNumClasses>0)
      {
         int iClassIx;

         for (iClassIx=1; iClassIx<=iNumClasses; ++iClassIx)
         {
            if (iClassIx>1)
               strcat(szBuffer, ",");
            SQX_GetDataFromIntVector(&oClasses, iClassIx, &iClass);
            sprintf(szBuffer+strlen(szBuffer), "%i", iClass);
         }
      }
      fprintf(pOut, "%-6s", szBuffer);
      if (!SQMWS_GetScaling(hWorkset, eBlock, ix, &oScaleInfo))
      {
         szErrString="SQMWS_GetScaling failed.";
         goto error_exit;
      }
      if (!SQMWS_GetShortScalingName(&oScaleInfo, &szName))
      {
         szErrString="SQMWS_GetShortScalingName failed.";
         goto error_exit;
      }
      fprintf(pOut, "%-10s", szName);
      if (!SQMWS_GetLongScalingName(&oScaleInfo, &szName))
      {
         szErrString="SQMWS_GetLongScalingName failed.";
         goto error_exit;
      }
      fprintf(pOut, "%s\n", szName);
   }
   fprintf(pOut, "\nName                     Transform      Long transform\n");
   /* List the variables, transforms */
   for (ix=1; ix<=iSize; ++ix)
   {
      if (!SQMWS_GetVariableName(hWorkset, eBlock, ix, 1, &szName))
      {
         szErrString="SQMWS_GetVariableName failed.";
         goto error_exit;
      }
      fprintf(pOut, "%-25s", szName);
      if (!SQMWS_GetTransform(hWorkset, eBlock, ix, &oTransformInfo))
      {
         szErrString="SQMWS_GetTransform failed.";
         goto error_exit;
      }
      if (!SQMWS_GetShortTransformName(&oTransformInfo, &szName))
      {
         szErrString="SQMWS_GetShortTransformName failed.";
         goto error_exit;
      }
      fprintf(pOut, "%-15s", szName);
      if (!SQMWS_GetLongTransformName(&oTransformInfo, &szName))
      {
         szErrString="SQMWS_GetLongTransformName failed.";
         goto error_exit;
      }
      fprintf(pOut, "%s\n", szName);
   }
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

static int PrintWorkset(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr)
{
   char  *szName;
   float       fMissingTolerance;
   char        *szErrString;

   if (!SQMWS_GetWorksetDescription(hWorkset, &szName))
   {
      szErrString="SQMWS_GetWorksetDescription failed.";
      goto error_exit;
   }
   fprintf(pOut, "Workset description = %s\n", szName);

   /* Print missing value tolerances */
   if (!SQMWS_GetMissingValueToleranceObs(hWorkset, &fMissingTolerance))
   {
      szErrString="SQMWS_GetMissingValueToleranceObs failed.";
      goto error_exit;
   }
   fprintf(pOut, "Missing value tolerance for observations = %g%%\n", fMissingTolerance*100);
   if (!SQMWS_GetMissingValueToleranceVar(hWorkset, &fMissingTolerance))
   {
      szErrString="SQMWS_GetMissingValueToleranceVar failed.";
      goto error_exit;
   }
   fprintf(pOut, "Missing value tolerance for variables    = %g%%\n", fMissingTolerance*100);

   if (!PrintBlock(hWorkset, SQM_BlockX, pOut, pErr))
      return 0;
   if (!PrintBlock(hWorkset, SQM_BlockY, pOut, pErr))
      return 0;
   if (!PrintObservations(hWorkset, pOut, pErr))
      return 0;
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

int Initialise(SQMRunner* pObj, FILE* pOut, FILE* pErr)
{
   char              *szProjectName = NULL;  /* Will contain the project name */
   const char        *szErrString;           /* String that will contain possible error. */
   int               iIsReducedUSP;          /* Will contain 1 if the project is a Reduced SIMCA-P project. */

   fprintf(pOut, "Path to usp file:        %s\n",pObj->mszUSPName);
   /* This is the file we want to use in SIMCA-QM */
   if (!SQX_AddProject(pObj->mszUSPName, 1, NULL, &pObj->mszProjID))
   {
      szErrString = "Could not open project. AddProject failed.";
      goto error_exit;
   }

   /*
   Get the project name
   */
   if (!SQX_GetProjectName(pObj->mszProjID, &szProjectName))
   {
      szErrString = "GetProjectName failed.";
      goto error_exit;
   }
   fprintf(pOut, "Project name:            %s\n",szProjectName);

   /* Check if the project is a Reduced SIMCA-P project.*/
   if (!SQX_GetIsReducedUSP(pObj->mszProjID, &iIsReducedUSP))
   {
      szErrString = "GetIsReducedUSP failed.";
      goto error_exit;
   }
   if(iIsReducedUSP == 1)
   {
      szErrString="The project is a reduced Simca project and can not be used to create new models.";
      goto error_exit;
   }

   /* make sure we are in modelling mode. */
   if (!SQM_ChangeMode(&pObj->mszProjID, SQM_Modeling))
   {
      szErrString="ChangeMode failes.";
      goto error_exit;
   }
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

int AddRemoveVariables(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr)
{
   const char        *szErrString;           /* String that will contain possible error. */

   /* Various variables that holds temporary values */
   int               ix;
   char       *szName;
   SQX_IntVector     oIndecies;
   SQX_IntVector     oLags;

   /* Add the square of the first variable in the primary dataset to the model. */
   SQX_InitIntVector(&oIndecies,2);
   SQX_SetDataInIntVector(&oIndecies, 1, 1);
   SQX_SetDataInIntVector(&oIndecies, 2, 1);
   SQX_InitIntVector(&oLags,2);
   SQX_SetDataInIntVector(&oLags, 1, 0);
   SQX_SetDataInIntVector(&oLags, 2, 0);
   if (!SQMWS_AddVariable(hWorkset, &oIndecies, &oLags, SQM_BlockX, &ix))
   {
      szErrString="SQMWS_AddVariable failed.";
      goto error_exit;
   }
   /* Get and print the name and index of the new variable. */
   if (!SQMWS_GetVariableName(hWorkset, SQM_BlockX, ix, 1, &szName))
   {
      szErrString="SQMWS_GetVariableName failed.";
      goto error_exit;
   }
   fprintf(pOut, "Variable %s added, index of the variable is %i\n", szName, ix);

   /* Add the first variable laged 2 steps */
   SQX_InitIntVector(&oIndecies,1);
   SQX_SetDataInIntVector(&oIndecies, 1, 1);
   SQX_InitIntVector(&oLags,1);
   SQX_SetDataInIntVector(&oLags, 1, 2);
   if (!SQMWS_AddVariable(hWorkset, &oIndecies, &oLags, SQM_BlockX, &ix))
   {
      szErrString="SQMWS_AddVariable failed.";
      goto error_exit;
   }
   /* Get and print the name and index of the new variable. */
   if (!SQMWS_GetVariableName(hWorkset, SQM_BlockX, ix, 1, &szName))
   {
      szErrString="SQMWS_GetVariableName failed.";
      goto error_exit;
   }
   fprintf(pOut, "Variable %s added, index of the variable is %i\n", szName, ix);

   /* Set the second variable as Y. If the variable is in the X-block, SQMWS_AddVariable */
   /* will remove it from the X-block before adding it to the Y-block. */
   SQX_InitIntVector(&oIndecies,1);
   SQX_SetDataInIntVector(&oIndecies, 1, 2);
   SQX_InitIntVector(&oLags,1);
   SQX_SetDataInIntVector(&oLags, 1, 0);
   if (!SQMWS_AddVariable(hWorkset, &oIndecies, &oLags, SQM_BlockY, &ix))
   {
      szErrString="SQMWS_AddVariable failed.";
      goto error_exit;
   }
   /* Get and print the name and index of the new variable. */
   if (!SQMWS_GetVariableName(hWorkset, SQM_BlockX, ix, 1, &szName))
   {
      szErrString="SQMWS_GetVariableName failed.";
      goto error_exit;
   }
   fprintf(pOut, "Variable %s added, index of the variable is %i\n", szName, ix);

   /* Remove the third variable in the dataset from the model */
   SQX_InitIntVector(&oIndecies,1);
   SQX_SetDataInIntVector(&oIndecies, 1, 3);
   SQX_InitIntVector(&oLags,1);
   SQX_SetDataInIntVector(&oLags, 1, 0);
   if (!SQMWS_GetVariableWorksetIndex(hWorkset, &oIndecies, &oLags, SQM_BlockX, &ix))
   {
      szErrString="SQMWS_GetVariableWorksetIndex failed.";
      goto error_exit;
   }
   /* Get the name of the variable */
   if (!SQMWS_GetVariableName(hWorkset, SQM_BlockX, ix, 1, &szName))
   {
      szErrString="SQMWS_GetVariableName failed.";
      goto error_exit;
   }
   fprintf(pOut, "Removing variable '%s' with index %i.\n", szName, ix);
   if (!SQMWS_RemoveVariable(hWorkset, SQM_BlockX, ix))
   {
      szErrString="SQMWS_RemoveVariable failed.";
      goto error_exit;
   }
   /* Set variable 7 to belong to class 1 and 2 */
   if (!SQMWS_SetVariableClass(hWorkset, SQM_BlockX, 7, 1))
   {
      szErrString="SQMWS_SetVariableClass failed.";
      goto error_exit;
   }
   if (!SQMWS_SetVariableClass(hWorkset, SQM_BlockX, 7, 2))
   {
      szErrString="SQMWS_SetVariableClass failed.";
      goto error_exit;
   }
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}

int PrintObservations(SQM_WorksetHandle hWorkset, FILE* pOut, FILE* pErr)
{
   char        *szErrString;
   int         iNumObservations;
   int         ix;
   int         iClassNo;
   char  *szName;

   if (!SQMWS_GetObservationSize(hWorkset, &iNumObservations))
   {
      szErrString="SQMWS_GetObservationSize failed.";
      goto error_exit;
   }
   fprintf(pOut, "\nThe workset has %i observations.\n", iNumObservations);
   fprintf(pOut, "Index Class Name\n");
   for (ix=1; ix<=iNumObservations; ++ix)
   {
      if (!SQMWS_GetObservationName(hWorkset, ix, 1, &szName))
      {
         szErrString="SQMWS_GetObservationName failed.";
         goto error_exit;
      }
      if (!SQMWS_GetObservationClass(hWorkset, ix, &iClassNo))
      {
         szErrString="SQMWS_GetObservationClass failed.";
         goto error_exit;
      }

      fprintf(pOut, "%-5i %-5i %s\n", ix, iClassNo, szName);
   }
   return 1;
error_exit:
   fputs(szErrString, pErr);
   return 0;
}
